/***************************************************************************************
 ***                                                                                 ***
 ***  Copyright (c) 2025, Lucid Vision Labs, Inc.                                    ***
 ***                                                                                 ***
 ***  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR     ***
 ***  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,       ***
 ***  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE    ***
 ***  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER         ***
 ***  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,  ***
 ***  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE  ***
 ***  SOFTWARE.                                                                      ***
 ***                                                                                 ***
 ***************************************************************************************/

 // Helios2 with PCL: Multithreading
 //    This example demonstrates displaying and segmenting a Helios2 camera point cloud 
 //    using the PCL library, and labeling the segments with dimension, depth, and outlines.
 //    This example uses multithreading, and is optimized for the Helios2+ camera.

 // LAST UPDATED: 2025-09-19


#define NOMINMAX
#define PCL_NO_PRECOMPILE 1
#undef max
#undef min

//#include "stdafx.h"
#include "ArenaApi.h"
#include "SaveApi.h"
#include <iostream>
#include <thread>
#include <pcl/filters/voxel_grid.h>
#include <pcl/segmentation/region_growing.h>
#include <iomanip> 
#include <pcl/visualization/cloud_viewer.h>
#include <pcl/features/normal_3d_omp.h>
#include <pcl/features/moment_of_inertia_estimation.h>
#include <pcl/filters/statistical_outlier_removal.h>
#include "cppturbo.hpp"
#include <mutex>
#include <queue>

#define TAB1 "  "
#define TAB2 "    "


#define SMOOTH_DEFAULT 7
#define CURVE_DEFAULT 0.2
#define RADIUS_DEFAULT 140
#define FLYPIX_DEFAULT 15
#define POINTSIZE_DEFAULT 1.0
#define CONFTHRESH_DEFAULT 12000
#define FONTSIZE_DEFAULT 80


float smoothnessThreshold = SMOOTH_DEFAULT;
float curvatureThreshold = CURVE_DEFAULT;
int confidenceThreshold = CONFTHRESH_DEFAULT;
int flyingPixel = FLYPIX_DEFAULT;
float radius = RADIUS_DEFAULT;
float point_size = POINTSIZE_DEFAULT;
int fontSize = FONTSIZE_DEFAULT;
bool revert = false;

bool onlyDisplayTopBB = true;
bool showAllBB = false;

void keyboardEventOccurred(const pcl::visualization::KeyboardEvent& event, void* viewer_void)
{
	if (event.keyDown())  // Key is pressed
	{
		//SMOOTHNESS
		if (event.getKeySym() == "Up") {
			smoothnessThreshold += 0.1;
		}
		else if (event.getKeySym() == "Right") {
			curvatureThreshold += 0.01;
		}
		//CURVATURE
		else if (event.getKeySym() == "Down") {
			smoothnessThreshold -= 0.1;
		}
		else if (event.getKeySym() == "Left") {
			curvatureThreshold -= 0.01;
		}
		//RADIUS
		else if (event.getKeySym() == "i") {
			radius += 1;
		}
		else if (event.getKeySym() == "k") {
			radius -= 1;
		}
		//FLYINGPIXEL
		else if (event.getKeySym() == "w") {
			flyingPixel += 1;
		}
		else if (event.getKeySym() == "s") {
			if (flyingPixel != 1) {
				flyingPixel -= 1;
			}
		}
		//CONFIDENCETHRESHOLD
		else if (event.getKeySym() == "4") {
			if (confidenceThreshold != 65400) {
				confidenceThreshold += 500;
			}
		}
		else if (event.getKeySym() == "1") {
			if (confidenceThreshold != 0) {
				confidenceThreshold -= 500;
			}
		}
		//REVERT CAMERA
		else if (event.getKeySym() == "v") {
			revert = true;
		}
		//POINTSIZE
		else if (event.getKeySym() == "6") {
			point_size += 0.5;
		}
		else if (event.getKeySym() == "3") {
			point_size -= 0.5;
		}
		//FONTSIZE
		else if (event.getKeySym() == "a") {
			if (fontSize != 20) {
				fontSize -= 10;
			}
		}
		else if (event.getKeySym() == "d") {
			if (fontSize != 150) {
				fontSize += 10;
			}
		}
		//DISPLAY 1
		else if (event.getKeySym() == "7") {
			onlyDisplayTopBB = true;
			showAllBB = false;
		}
		//DISPLAY 2
		else if (event.getKeySym() == "8") {
			onlyDisplayTopBB = false;
			showAllBB = true;
		}
		//DISPLAY 3
		else if (event.getKeySym() == "9") {
			onlyDisplayTopBB = false;
			showAllBB = false;
		}
		//BACK TO DEFAULT
		else if (event.getKeySym() == "z") {
			smoothnessThreshold = SMOOTH_DEFAULT;
			curvatureThreshold = CURVE_DEFAULT;
			confidenceThreshold = CONFTHRESH_DEFAULT;
			flyingPixel = FLYPIX_DEFAULT;
			radius = RADIUS_DEFAULT;
			point_size = POINTSIZE_DEFAULT;
			fontSize = FONTSIZE_DEFAULT;
		}
	}
}


// store:
// 1. bounding box (bb)
//		a. length
//		b. width
//		c. height
// 2. bb position (centroid)
// 3. bb quaternion (based on rotational matrix)
// 4. bb (centroid) depth
// 5. text associated with the bb
// 6. position of text location
struct bbInfo {
	int length;
	int width;
	int height;
	Eigen::Vector3f position;
	Eigen::Quaternionf quat;

	int depth;
	std::string text;
	pcl::PointXYZ text_position;
};

// store:
// 1. PCL point cloud of the segment
// 2. centroid of the segment
struct segmentInfo {
	pcl::PointCloud<pcl::PointXYZRGB>::Ptr cloud;
	Eigen::Vector4f centroid;
};

// global variables for multithreading
std::mutex mtx;
std::condition_variable condVar;
bool newBoundingBoxesAvailable = false;
bool newPointCloudAvailable = false;
bool viewerClosed = false;
pcl::PointCloud<pcl::PointXYZRGB>::Ptr sharedCloud(new pcl::PointCloud<pcl::PointXYZRGB>);
std::vector <std::pair <bool, bbInfo>> sharedBBs;

// takes a segmentInfo object, and
// returns a bbInfo object. the bbInfo.text is not filled here - it is filled under after further
// processing of the cloud, and user-set settings later in the code
bbInfo getBoundingBoxInfo(segmentInfo segment) {

	Eigen::Vector3f position;
	Eigen::Vector3f positionOBB;
	Eigen::Vector3f dimensionsOBB;
	Eigen::Matrix3f rotationalMatrixOBB;

	pcl::computeCentroidAndOBB(*segment.cloud, position, positionOBB, dimensionsOBB, rotationalMatrixOBB);

	int length = dimensionsOBB[0]; // these are the BB dimensions in PCL native coordinates
	int width = dimensionsOBB[1];
	int height = dimensionsOBB[2];

	int depth = segment.centroid[2] / 10; // in cm

	//initialize bbInfo object to popoulate with bounding box data
	bbInfo myBB;

	myBB.length = length;
	myBB.width = width;
	myBB.height = height;

	myBB.position << positionOBB[0], positionOBB[1], positionOBB[2];

	myBB.quat = Eigen::Quaternionf(rotationalMatrixOBB);

	myBB.depth = depth;

	myBB.text_position.x = positionOBB[0] + length / 2;
	myBB.text_position.y = positionOBB[1];
	myBB.text_position.z = positionOBB[2] - 200;


	return myBB;

}

// takes an unsegmented, filtered point cloud, and segments it
// returns a segmented point cloud (with PCL's own coloring), and also populates the other parameter (that points
// to a vector of pcl::PointIndices) with the indices of all the segments produced
pcl::PointCloud<pcl::PointXYZRGB>::Ptr segmentAndColor(pcl::PointCloud<pcl::PointXYZRGB>::Ptr cloudUnsegmented,
	std::vector <pcl::PointIndices>* clusters) {

	pcl::PointXYZRGB point;
	pcl::search::Search<pcl::PointXYZRGB>::Ptr tree(new pcl::search::KdTree<pcl::PointXYZRGB>);
	pcl::PointCloud <pcl::Normal>::Ptr normals(new pcl::PointCloud <pcl::Normal>);
	pcl::NormalEstimation<pcl::PointXYZRGB, pcl::Normal> normalEstimator;

	pcl::IndicesPtr indices(new std::vector <int>);

	pcl::RegionGrowing<pcl::PointXYZRGB, pcl::Normal> reg;


	normalEstimator.setSearchMethod(tree);
	normalEstimator.setInputCloud(cloudUnsegmented);

	normalEstimator.setRadiusSearch(radius);
	normalEstimator.compute(*normals);

	pcl::removeNaNFromPointCloud(*cloudUnsegmented, *indices);

	reg.setMinClusterSize(500);
	reg.setMaxClusterSize(5000);
	reg.setSearchMethod(tree);
	reg.setNumberOfNeighbours(5);
	reg.setInputCloud(cloudUnsegmented);
	reg.setIndices(indices);
	reg.setInputNormals(normals);
	reg.setSmoothnessThreshold(smoothnessThreshold / 180.0 * M_PI);
	reg.setCurvatureThreshold(curvatureThreshold);

	reg.extract(*clusters);

	return reg.getColoredCloud();
}

// NO VIEWER METHODS IN THIS THREAD
void pclExampleProcessingThread(double& xyz_scale_mm, double& x_offset_mm, double& y_offset_mm, double& z_offset_mm,
	double& tof_to_xyz_conversion_offset) {

	// initialize PCL point cloud to store PCL's segmented output (with PCL colors)
	pcl::PointCloud<pcl::PointXYZRGB>::Ptr cloudColoredBySegments(new pcl::PointCloud<pcl::PointXYZRGB>);

	// initialize PCL point indices to store output of PCL's Region Based Growing Segmentation. Note: the output
	// is a set of INDICES of the segmented clusters, not the clusters' points themselves. These indices will need to be used 
	// on the filtered cloud (that the segmentation was done on) to obtain the corresponding segments
	std::vector <pcl::PointIndices> segmentedClusters;

	// initialize PCL point cloud to temporarily store our segments as we iterate through 'segmentedClusters'
	pcl::PointCloud<pcl::PointXYZRGB>::Ptr tempCluster(new pcl::PointCloud<pcl::PointXYZRGB>);

	// initialize PCL point cloud to store the Helios native cloud shared from 'pclExampleVisualizationThread'
	pcl::PointCloud<pcl::PointXYZRGB>::Ptr localCloud(new pcl::PointCloud<pcl::PointXYZRGB>);

	// initialize a PCL point to individually add Helios points (with their co-ordinates and color)
	pcl::PointXYZRGB point;

	// initialize point to store centroid of a cluster, in native PCL coordinates)
	Eigen::Vector4f centroidNative;

	// initialize point to store centroid of a cluster, in real-world coordinates)
	Eigen::Vector4f centroidReal;



	while (!viewerClosed) {

		// LOCK thread and wait for a new point cloud, and update shared boolean
		std::unique_lock<std::mutex> lock(mtx);
		condVar.wait(lock, [] { return newPointCloudAvailable; });

		*localCloud = *sharedCloud;
		newPointCloudAvailable = false;
		// UNLOCK
		lock.unlock();

		pcl::PointCloud<pcl::PointXYZRGB>::Ptr cloudFiltered(new pcl::PointCloud<pcl::PointXYZRGB>);
		pcl::VoxelGrid<pcl::PointXYZRGB> vg;
		vg.setInputCloud(localCloud);
		vg.setLeafSize(40.0f, 40.0f, 40.0f);
		vg.filter(*cloudFiltered);
		cloudColoredBySegments = segmentAndColor(cloudFiltered, &segmentedClusters);

		// initialize vector to store segments (with their centroids) as we iterate through 'segmentedClusters'
		std::vector < segmentInfo > currentSegments;

		// iterate through 'segmentedClusters'
		for (const auto& cluster : segmentedClusters) {

			for (const auto& idx : cluster.indices) {

				tempCluster->push_back((*cloudFiltered)[idx]);
			}

			pcl::compute3DCentroid(*tempCluster, centroidNative);

			centroidReal[0] = centroidNative[0] * xyz_scale_mm + x_offset_mm;
			centroidReal[1] = centroidNative[1] * xyz_scale_mm + y_offset_mm;
			centroidReal[2] = centroidNative[2] * xyz_scale_mm + z_offset_mm + tof_to_xyz_conversion_offset;

			// calculate how far the centroid of cluster from camera front
			float distance_to_origin = std::sqrt(centroidReal[0] * centroidReal[0] + centroidReal[1] * centroidReal[1] + centroidReal[2] * centroidReal[2]);

			//std::cout << "distance: " << distance_to_origin << ", centroidReal[z]: " << centroidReal[2] << std::endl;

			// filter out clusters farther away from user-set threshold
			if (distance_to_origin > 1800) {
				tempCluster->clear();
				continue;
			}

			segmentInfo newSegment;
			newSegment.cloud.reset(new pcl::PointCloud<pcl::PointXYZRGB>);
			*newSegment.cloud += *tempCluster;
			newSegment.centroid = centroidReal;
			currentSegments.push_back({ newSegment });

			tempCluster->clear();

		}

		// sort the segments in ascending order based on the centroid depth.
		// this is done to prioritize segments with a smaller depth over those with a larger depth, since depth
		// used to ignore segments too close in proximity when adding bounding boxes (see below)
		std::sort(currentSegments.begin(), currentSegments.end(), [](const segmentInfo& a, const segmentInfo& b) {
			return (a.centroid[2]) < (b.centroid[2]);
			});

		{
			// LOCK thread to clear sharedBBs
			std::lock_guard<std::mutex> lock(mtx);
			sharedBBs.clear();
			// UNLOCK
		}

		std::vector<std::pair<float, float>> positionTracker;
		int count = 0;
		bool ignore;
		for (auto& segment : currentSegments) {
			ignore = false;

			bool onlyText = false;
			bbInfo segmentBB = getBoundingBoxInfo(segment);

			for (int b = 0; b < count; b++) {
				float difference_x = std::abs(segmentBB.position[0] - positionTracker[b].first);
				float difference_y = std::abs(segmentBB.position[1] - positionTracker[b].second);

				//std::cout << "difference_x: " << difference_x << ", differenc_y: " << difference_y << std::endl;

				// ignore segments for bounding boxes with user-set proximity threshold (set to 1100 here)
				if (difference_x < 1100 && difference_y < 1100) {
					//std::cout << "overalpping" << std::endl;
					ignore = true;
					break;
				}
			}

			positionTracker.push_back({ segmentBB.position[0], segmentBB.position[1] });

			std::stringstream ss;
			//std::string text;

			if (ignore == false) {
				// check display options:
				// if 'showAllBB' is true, show bounding box + dimensions + depth for all segments,
				// if 'onlyDisplayTopBB' is true, show bounding box + dimensions + depth for topmost segment, and only depth for other segments
				// if neither is true, only show depth for all segments
				if (showAllBB || (onlyDisplayTopBB && count == 0)) {

					ss << int(segmentBB.length / 40) << "cm x " << int(segmentBB.width / 40) << "cm, depth: " << segmentBB.depth << "cm"; // length and width are now in cm for viewer display

				}

				else {

					onlyText = true;
					ss << "depth: " << segmentBB.depth << "cm"; // only display the depth
				}

				segmentBB.text = ss.str();

				{
					// LOCK thread to populate sharedBBs
					std::lock_guard<std::mutex> lock(mtx);
					sharedBBs.push_back({ onlyText, segmentBB });
					// UNLOCK
				}
			}

			count++;
		}

		positionTracker.clear();

		{
			// LOCK thread to update shared boolean
			std::lock_guard<std::mutex> lock(mtx);
			newBoundingBoxesAvailable = true;
			// UNLOCK
		}

	}

}

void pclExampleVisualizationThread(Arena::IDevice* pHeliosDevice, size_t& width, size_t& height, double& xyz_scale_mm, double& z_offset_mm,
	double& tof_to_xyz_conversion_offset) {

	pcl::visualization::PCLVisualizer::Ptr viewer(new pcl::visualization::PCLVisualizer("HTP003S-001"));

	// initialize the PCL point cloud to populate the Helios native point cloud with
	pcl::PointCloud<pcl::PointXYZRGB>::Ptr cloudNative(new pcl::PointCloud<pcl::PointXYZRGB>);

	// initialize a PCL point to individually add Helios points (with their co-ordinates and color)
	pcl::PointXYZRGB point;

	// this will store color information for each point
	turbo::Color3f floats;

	// add point cloud to the viewer
	viewer->addPointCloud(cloudNative);

	// the default camera position of the viewer on launch can be off target. to avoid zooming in / out, translating, 
	// or rotating the point cloud on every launch, the following values were chosen to have the optimal view of the target.
	// (in the real world, the camera here is centered about 1-2m above the target, looking down at it)
	viewer->setCameraPosition(33600, 31200, -8000, 32700, 32300, 5692, 0, 1, 0);

	viewer->setBackgroundColor(0.1, 0.1, 0.1);

	// register this callback keep checking for keys set in the code that can adjust different ArenaSDK / PCL parameters. 
	// see 'keyboardEventOccured' above
	viewer->registerKeyboardCallback(keyboardEventOccurred, (void*)viewer.get());

	viewer->setShowFPS(false);

	int iteration = 0; // for bounding boxes

	pHeliosDevice->StartStream();

	// start the viewer
	while (!viewer->wasStopped())
	{
		//std::cout << iteration << std::endl;

		Arena::SetNodeValue<int64_t>(pHeliosDevice->GetNodeMap(), "Scan3dFlyingPixelsDistanceThreshold", flyingPixel);
		Arena::SetNodeValue<int64_t>(pHeliosDevice->GetNodeMap(), "Scan3dConfidenceThresholdMin", confidenceThreshold);

		Arena::IImage* pHeliosImage = pHeliosDevice->GetImage(2000);

		width = pHeliosImage->GetWidth();
		height = pHeliosImage->GetHeight();

		const uint16_t* input_data = reinterpret_cast<const uint16_t*>(pHeliosImage->GetData());

		for (unsigned int ir = 0; ir < height; ++ir)
		{
			for (unsigned int ic = 0; ic < width; ++ic)
			{
				//Get unsigned 16 bit values for X,Y,Z coordinates
				unsigned short x_u16 = (float)(input_data[0]);
				unsigned short y_u16 = (float)(input_data[1]);
				unsigned short z_u16 = (float)(input_data[2]);
				//unsigned short intensity_u16 = (float)(input_data[3]);

				// calculate real-world depth using xyz_scale_mm and set point's color according to depth
				float depth = z_u16 * xyz_scale_mm - tof_to_xyz_conversion_offset;
				turbo::GetColor((depth - 300) / 1600, floats);

				point.x = (x_u16);
				point.y = (y_u16);
				point.z = (z_u16);

				point.r = static_cast<uint8_t>(floats.r * 255); // Red
				point.g = static_cast<uint8_t>(floats.g * 255); // Green
				point.b = static_cast<uint8_t>(floats.b * 255); // Blue

				cloudNative->push_back(point);

				input_data += 4;
			}
		}

		viewer->setPointCloudRenderingProperties(pcl::visualization::PCL_VISUALIZER_POINT_SIZE, point_size);

		{
			// LOCK thread to transfer over new point cloud, and update shared boolean
			std::lock_guard<std::mutex> lock(mtx);
			*sharedCloud = *cloudNative;
			newPointCloudAvailable = true;
		}
		condVar.notify_one();

		viewer->updatePointCloud(cloudNative);

		{
			// LOCK thread and add bb's to viewer
			std::lock_guard<std::mutex> lock(mtx);

			if (newBoundingBoxesAvailable) {

				newBoundingBoxesAvailable = false;
				viewer->removeAllShapes();

				int counter = iteration + 20; // 20 is arbitrary here - this assumes total number of boxes are less than 20
				for (int i = 0; i < sharedBBs.size(); i++) {

					viewer->addText3D(sharedBBs[i].second.text, sharedBBs[i].second.text_position, fontSize, 1.0, 1.0, 1.0, "text" + std::to_string(counter));

					if (!sharedBBs[i].first) {
						viewer->addCube(sharedBBs[i].second.position, sharedBBs[i].second.quat, sharedBBs[i].second.length, sharedBBs[i].second.width, sharedBBs[i].second.height, "OBB" + std::to_string(counter));
						viewer->setShapeRenderingProperties(pcl::visualization::PCL_VISUALIZER_REPRESENTATION, pcl::visualization::PCL_VISUALIZER_REPRESENTATION_WIREFRAME, "OBB" + std::to_string(counter));
					}

					counter++;
				}
			}
			// UNLOCK
		}

		if (revert == true) {
			viewer->setCameraPosition(33600, 31200, -8000, 32700, 32300, 5692, 0, 1, 0);
			revert = false;
		}

		std::cout << "smooth: " << smoothnessThreshold << ", " << "curve: " << curvatureThreshold << ", " << "radius: " << radius << ", " << "confThresh: " << confidenceThreshold << ", " << "flyPix : " << flyingPixel << std::endl;

		viewer->spinOnce();
		cloudNative->clear();

		// display the current camera position in terminal to determine how to adjust the camera in the viewer
		/*pcl::visualization::Camera camera;

		viewer->getCameraParameters(camera);

		std::cout << "Camera position: ("
		<< camera.pos[0] << ", "
		<< camera.pos[1] << ", "
		<< camera.pos[2] << ", "
		<< camera.focal[0] << ", "
		<< camera.focal[1] << ", "
		<< camera.focal[2] << ", "
		<< camera.view[0] << ", "
		<< camera.view[1] << ", "
		<< camera.view[2] << ") " << std::endl;*/

		pHeliosDevice->RequeueBuffer(pHeliosImage);

		iteration++;

	}

	{
		// LOCK thread to update shared boolean
		std::lock_guard<std::mutex> lock(mtx);
		viewerClosed = true;
		// UNLOCK
	}

}

void pclExampleMain(Arena::IDevice* pHeliosDevice, size_t& width, size_t& height, double& xyz_scale_mm, double& x_offset_mm, double& y_offset_mm, double& z_offset_mm,
	double& tof_to_xyz_conversion_offset) {
	// obtain the scale from native point cloud co-ordinates to mm
	// obtain the x, y, z offsets
	// obtain the conversion offset
	GenApi::INodeMap* node_map = pHeliosDevice->GetNodeMap();
	xyz_scale_mm = Arena::GetNodeValue<double>(node_map, "Scan3dCoordinateScale");
	Arena::SetNodeValue<GenICam::gcstring>(node_map, "Scan3dCoordinateSelector", "CoordinateA");
	x_offset_mm = Arena::GetNodeValue<double>(node_map, "Scan3dCoordinateOffset");
	Arena::SetNodeValue<GenICam::gcstring>(node_map, "Scan3dCoordinateSelector", "CoordinateB");
	y_offset_mm = Arena::GetNodeValue<double>(node_map, "Scan3dCoordinateOffset");
	Arena::SetNodeValue<GenICam::gcstring>(node_map, "Scan3dCoordinateSelector", "CoordinateC");
	z_offset_mm = Arena::GetNodeValue<double>(node_map, "Scan3dCoordinateOffset");
	tof_to_xyz_conversion_offset = Arena::GetNodeValue<double>(node_map, "TofXyzConversionOffsetZ");

	Arena::SetNodeValue<GenICam::gcstring>(node_map, "Scan3dHDRMode", "StandardHDR");

	Arena::SetNodeValue<int64_t>(pHeliosDevice->GetNodeMap(), "Scan3dConfidenceThresholdMin", confidenceThreshold); // was 7000

	Arena::SetNodeValue<bool>(pHeliosDevice->GetNodeMap(), "Scan3dFlyingPixelsRemovalEnable", true); // was 7000

	std::thread visualization(pclExampleVisualizationThread, pHeliosDevice, std::ref(width), std::ref(height), std::ref(xyz_scale_mm), std::ref(z_offset_mm), std::ref(tof_to_xyz_conversion_offset));
	std::thread processing(pclExampleProcessingThread, std::ref(xyz_scale_mm), std::ref(x_offset_mm), std::ref(y_offset_mm), std::ref(z_offset_mm), std::ref(tof_to_xyz_conversion_offset));

	// Join threads (this will block until the threads finish)
	processing.join();
	visualization.join();

	pHeliosDevice->StopStream();
}


// -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
// -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
// -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
// -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
// -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------



Arena::DeviceInfo SelectDevice(std::vector<Arena::DeviceInfo>& deviceInfos)
{
	if (deviceInfos.size() == 1)
	{
		std::cout << "\n" << TAB1 << "Only one device detected: " << deviceInfos[0].ModelName() << TAB1 << deviceInfos[0].SerialNumber() << TAB1 << deviceInfos[0].IpAddressStr() << ".\n";
		std::cout << TAB1 << "Automatically selecting this device.\n";
		return deviceInfos[0];
	}

	std::cout << "\nSelect device:\n";
	for (size_t i = 0; i < deviceInfos.size(); i++)
	{
		std::cout << TAB1 << i + 1 << ". " << deviceInfos[i].ModelName() << TAB1 << deviceInfos[i].SerialNumber() << TAB1 << deviceInfos[i].IpAddressStr() << "\n";
	}
	size_t selection = 0;

	do
	{
		std::cout << TAB1 << "Make selection (1-" << deviceInfos.size() << "): ";
		std::cin >> selection;

		if (std::cin.fail())
		{
			std::cin.clear();
			while (std::cin.get() != '\n');
			std::cout << TAB1 << "Invalid input. Please enter a number.\n";
		}
		else if (selection <= 0 || selection > deviceInfos.size())
		{
			std::cout << TAB1 << "Invalid device selected. Please select a device in the range (1-" << deviceInfos.size() << ").\n";
		}

	} while (selection <= 0 || selection > deviceInfos.size());

	return deviceInfos[selection - 1];
}

// =-=-=-=-=-=-=-=-=-
// =- PREPARATION -=-
// =- & CLEAN UP =-=-
// =-=-=-=-=-=-=-=-=-

int main()
{
	// flag to track when an exception has been thrown
	bool exceptionThrown = false;

	std::cout << "Cpp_HTP_PCL_4_Multithreading";

	try
	{
		// prepare example
		Arena::ISystem* pSystem = Arena::OpenSystem();
		pSystem->UpdateDevices(100);
		std::vector<Arena::DeviceInfo> deviceInfos = pSystem->GetDevices();
		if (deviceInfos.size() == 0)
		{
			std::cout << "\nNo camera connected\nPress enter to complete\n";
			std::getchar();
			return 0;
		}
		Arena::DeviceInfo selectedDeviceInfo = SelectDevice(deviceInfos);
		Arena::IDevice* pDevice = pSystem->CreateDevice(selectedDeviceInfo);

		// enable stream auto negotiate packet size
		Arena::SetNodeValue<bool>(pDevice->GetTLStreamNodeMap(), "StreamAutoNegotiatePacketSize", true);

		// enable stream packet resend
		Arena::SetNodeValue<bool>(pDevice->GetTLStreamNodeMap(), "StreamPacketResendEnable", true);

		// enable flying pixels removal 
		Arena::SetNodeValue<bool>(pDevice->GetNodeMap(), "Scan3dFlyingPixelsRemovalEnable", true);

		// enable Standard HDR
		Arena::SetNodeValue<GenICam::gcstring>(pDevice->GetNodeMap(), "Scan3dHDRMode", "StandardHDR");

		std::cout << "Commence example\n\n";

		/*cv::Mat image_XYZ_mm;*/
		size_t width = 0;
		size_t height = 0;
		double scale;
		double offset_x;
		double offset_y;
		double offset_z;
		double offset_tof_to_xyz;

		pclExampleMain(
			pDevice,
			width,
			height,
			scale,
			offset_x,
			offset_y,
			offset_z,
			offset_tof_to_xyz);

		std::cout << "\nExample complete\n";

		// clean up example
		pSystem->DestroyDevice(pDevice);
		Arena::CloseSystem(pSystem);
	}
	catch (GenICam::GenericException& ge)
	{
		std::cout << "\nGenICam exception thrown: " << ge.what() << "\n";
		exceptionThrown = true;
	}
	catch (std::exception& ex)
	{
		std::cout << "\nStandard exception thrown: " << ex.what() << "\n";
		exceptionThrown = true;
	}
	catch (...)
	{
		std::cout << "\nUnexpected exception thrown\n";
		exceptionThrown = true;
	}

	std::cout << "Press enter to complete\n";
	std::cin.ignore();
	std::getchar();

	if (exceptionThrown)
		return -1;
	else
		return 0;
}
