/***************************************************************************************
 ***                                                                                 ***
 ***  Copyright (c) 2025, Lucid Vision Labs, Inc.                                    ***
 ***                                                                                 ***
 ***  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR     ***
 ***  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,       ***
 ***  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE    ***
 ***  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER         ***
 ***  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,  ***
 ***  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE  ***
 ***  SOFTWARE.                                                                      ***
 ***                                                                                 ***
 ***************************************************************************************/

 // Helios2 with PCL: Segmentation
 //    This example demonstrates displaying a Helios2 camera point cloud using the
 //    PCL library, and segmenting the point cloud into separate surfaces.
 //    This example is optimized for the Helios2+ camera.

 // LAST UPDATED: 2025-09-19


#define NOMINMAX
#define PCL_NO_PRECOMPILE 1
#undef max
#undef min

//#include "stdafx.h"
#include "ArenaApi.h"
#include "SaveApi.h"
#include <iostream>
#include <thread>
#include <pcl/filters/voxel_grid.h>
#include <pcl/segmentation/region_growing.h>
#include <iomanip> 
#include <pcl/visualization/cloud_viewer.h>
#include <pcl/features/normal_3d_omp.h>
#include <pcl/features/moment_of_inertia_estimation.h>
#include <pcl/filters/statistical_outlier_removal.h>
#include "cppturbo.hpp"
#include <mutex>
#include <queue>

#define TAB1 "  "
#define TAB2 "    "


#define SMOOTH_DEFAULT 7
#define CURVE_DEFAULT 0.2
#define RADIUS_DEFAULT 140
#define FLYPIX_DEFAULT 15
#define POINTSIZE_DEFAULT 1.0
#define CONFTHRESH_DEFAULT 12000


float smoothnessThreshold = SMOOTH_DEFAULT;
float curvatureThreshold = CURVE_DEFAULT;
int confidenceThreshold = CONFTHRESH_DEFAULT;
int flyingPixel = FLYPIX_DEFAULT;
float radius = RADIUS_DEFAULT;
float point_size = POINTSIZE_DEFAULT;
bool revert = false;

void keyboardEventOccurred(const pcl::visualization::KeyboardEvent& event, void* viewer_void)
{
	if (event.keyDown())  // Key is pressed
	{
		//SMOOTHNESS
		if (event.getKeySym() == "Up") {
			smoothnessThreshold += 0.1;
		}
		else if (event.getKeySym() == "Right") {
			curvatureThreshold += 0.01;
		}
		//CURVATURE
		else if (event.getKeySym() == "Down") {
			smoothnessThreshold -= 0.1;
		}
		else if (event.getKeySym() == "Left") {
			curvatureThreshold -= 0.01;
		}
		//FLYINGPIXEL
		else if (event.getKeySym() == "w") {
			flyingPixel += 1;
		}
		else if (event.getKeySym() == "s") {
			if (flyingPixel != 1) {
				flyingPixel -= 1;
			}
		}
		//REVERT CAMERA
		else if (event.getKeySym() == "v") {
			revert = true;
		}
		//RADIUS
		else if (event.getKeySym() == "i") {
			radius += 1;
		}
		else if (event.getKeySym() == "k") {
			radius -= 1;
		}
		//POINTSIZE
		else if (event.getKeySym() == "6") {
			point_size += 0.5;
		}
		else if (event.getKeySym() == "3") {
			point_size -= 0.5;
		}
		//CONFIDENCETHRESHOLD
		else if (event.getKeySym() == "4") {
			if (confidenceThreshold != 65400) {
				confidenceThreshold += 500;
			}
		}
		else if (event.getKeySym() == "1") {
			if (confidenceThreshold != 0) {
				confidenceThreshold -= 500;
			}
		}
		//BACK TO DEFAULT
		else if (event.getKeySym() == "z") {
			smoothnessThreshold = SMOOTH_DEFAULT;
			curvatureThreshold = CURVE_DEFAULT;
			confidenceThreshold = CONFTHRESH_DEFAULT;
			flyingPixel = FLYPIX_DEFAULT;
			radius = RADIUS_DEFAULT;
			point_size = POINTSIZE_DEFAULT;
		}
	}
}


// takes an unsegmented, filtered point cloud, and segments it
// returns a segmented point cloud (with PCL's own coloring), and also populates the other parameter (that points
// to a vector of pcl::PointIndices) with the indices of all the segments produced
pcl::PointCloud<pcl::PointXYZRGB>::Ptr segmentAndColor(pcl::PointCloud<pcl::PointXYZRGB>::Ptr cloudUnsegmented,
	std::vector <pcl::PointIndices>* clusters) {

	pcl::PointXYZRGB point;
	pcl::search::Search<pcl::PointXYZRGB>::Ptr tree(new pcl::search::KdTree<pcl::PointXYZRGB>);
	pcl::PointCloud <pcl::Normal>::Ptr normals(new pcl::PointCloud <pcl::Normal>);
	pcl::NormalEstimation<pcl::PointXYZRGB, pcl::Normal> normalEstimator;

	pcl::IndicesPtr indices(new std::vector <int>);

	pcl::RegionGrowing<pcl::PointXYZRGB, pcl::Normal> reg;


	normalEstimator.setSearchMethod(tree);
	normalEstimator.setInputCloud(cloudUnsegmented);

	normalEstimator.setRadiusSearch(radius);
	normalEstimator.compute(*normals);

	pcl::removeNaNFromPointCloud(*cloudUnsegmented, *indices);

	reg.setMinClusterSize(500);
	reg.setMaxClusterSize(5000);
	reg.setSearchMethod(tree);
	reg.setNumberOfNeighbours(5);
	reg.setInputCloud(cloudUnsegmented);
	reg.setIndices(indices);
	reg.setInputNormals(normals);
	reg.setSmoothnessThreshold(smoothnessThreshold / 180.0 * M_PI);
	reg.setCurvatureThreshold(curvatureThreshold);

	reg.extract(*clusters);

	return reg.getColoredCloud();
}


void pclExample(Arena::IDevice* pHeliosDevice, size_t& width, size_t& height, double& xyz_scale_mm, double& x_offset_mm, double& y_offset_mm, double& z_offset_mm,
	double& tof_to_xyz_conversion_offset)
{
	// obtain the scale from native point cloud co-ordinates to mm
	// obtain the x, y, z offsets
	// obtain the conversion offset
	GenApi::INodeMap* node_map = pHeliosDevice->GetNodeMap();
	xyz_scale_mm = Arena::GetNodeValue<double>(node_map, "Scan3dCoordinateScale");
	Arena::SetNodeValue<GenICam::gcstring>(node_map, "Scan3dCoordinateSelector", "CoordinateA");
	x_offset_mm = Arena::GetNodeValue<double>(node_map, "Scan3dCoordinateOffset");
	Arena::SetNodeValue<GenICam::gcstring>(node_map, "Scan3dCoordinateSelector", "CoordinateB");
	y_offset_mm = Arena::GetNodeValue<double>(node_map, "Scan3dCoordinateOffset");
	Arena::SetNodeValue<GenICam::gcstring>(node_map, "Scan3dCoordinateSelector", "CoordinateC");
	z_offset_mm = Arena::GetNodeValue<double>(node_map, "Scan3dCoordinateOffset");
	tof_to_xyz_conversion_offset = Arena::GetNodeValue<double>(node_map, "TofXyzConversionOffsetZ");

	// initialize the PCL Visualizer (called 'viewer' in this code throughout)
	pcl::visualization::PCLVisualizer::Ptr viewer(new pcl::visualization::PCLVisualizer("HTP003S-001"));

	// initialize the PCL point cloud to populate the Helios native point cloud with
	pcl::PointCloud<pcl::PointXYZRGB>::Ptr cloudNative(new pcl::PointCloud<pcl::PointXYZRGB>);

	// initialize PCL point cloud to store PCL's segmented output (with PCL colors)
	pcl::PointCloud<pcl::PointXYZRGB>::Ptr cloudColoredBySegments(new pcl::PointCloud<pcl::PointXYZRGB>);

	// initialize PCL point indices to store output of PCL's Region Based Growin Segmentation. Note: the output
	// is a set of INDICES of the segmented clusters, not the clusters themselves. These indices will need to be used 
	// on the filtered cloud (that the segmentation was done on) to obtain the corresponding segments
	std::vector <pcl::PointIndices> segmentedClusters;

	// initialize a PCL point to individually add Helios points (with their co-ordinates and color)
	pcl::PointXYZRGB point;

	// add point cloud to the viewer
	//viewer->addPointCloud(cloudNative);
	viewer->addPointCloud(cloudColoredBySegments);

	// the default camera position of the viewer on launch can be off target. to avoid zooming in / out, translating, 
	// or rotating the point cloud on every launch, the following values were chosen to have the optimal view of the target.
	// (in the real world, the camera here is centered about 1-2m above the target, looking down at it)
	viewer->setCameraPosition(33600, 31200, -8000, 32700, 32300, 5692, 0, 1, 0);

	viewer->setBackgroundColor(0.1, 0.1, 0.1);

	// register this callback keep checking for keys set in the code that can adjust different ArenaSDK / PCL parameters. 
	// see 'keyboardEventOccured' above
	viewer->registerKeyboardCallback(keyboardEventOccurred, (void*)viewer.get());

	viewer->setShowFPS(false);

	// this will store color information for each point
	turbo::Color3f floats;

	pHeliosDevice->StartStream();

	// start the viewer
	while (!viewer->wasStopped())
	{
		Arena::SetNodeValue<int64_t>(pHeliosDevice->GetNodeMap(), "Scan3dFlyingPixelsDistanceThreshold", flyingPixel);
		Arena::SetNodeValue<int64_t>(pHeliosDevice->GetNodeMap(), "Scan3dConfidenceThresholdMin", confidenceThreshold);

		Arena::IImage* pHeliosImage = pHeliosDevice->GetImage(2000);

		width = pHeliosImage->GetWidth();
		height = pHeliosImage->GetHeight();

		const uint16_t* input_data = reinterpret_cast<const uint16_t*>(pHeliosImage->GetData());

		for (unsigned int ir = 0; ir < height; ++ir)
		{
			for (unsigned int ic = 0; ic < width; ++ic)
			{
				//Get unsigned 16 bit values for X,Y,Z coordinates
				unsigned short x_u16 = (float)(input_data[0]);
				unsigned short y_u16 = (float)(input_data[1]);
				unsigned short z_u16 = (float)(input_data[2]);
				//unsigned short intensity_u16 = (float)(input_data[3]);

				// calculate real-world depth using xyz_scale_mm and set point's color according to depth
				float depth = z_u16 * xyz_scale_mm - tof_to_xyz_conversion_offset;
				turbo::GetColor((depth - 300) / 1600, floats);


				point.x = (x_u16);
				point.y = (y_u16);
				point.z = (z_u16);

				point.r = static_cast<uint8_t>(floats.r * 255); // Red
				point.g = static_cast<uint8_t>(floats.g * 255); // Green
				point.b = static_cast<uint8_t>(floats.b * 255); // Blue

				cloudNative->push_back(point);

				input_data += 4;
			}
		}

		// filter cloudNative to get cloudFiltered
		// segment cloudFiltered
		pcl::PointCloud<pcl::PointXYZRGB>::Ptr cloudFiltered(new pcl::PointCloud<pcl::PointXYZRGB>);
		pcl::VoxelGrid<pcl::PointXYZRGB> vg;
		vg.setInputCloud(cloudNative);
		vg.setLeafSize(40.0f, 40.0f, 40.0f);
		vg.filter(*cloudFiltered);
		cloudColoredBySegments = segmentAndColor(cloudFiltered, &segmentedClusters);

		// revert camera position to original
		if (revert == true) {
			viewer->setCameraPosition(33600, 31200, -8000, 32700, 32300, 5692, 0, 1, 0);
			revert = false;
		}

		// adjust point size - keyboard callback
		viewer->setPointCloudRenderingProperties(pcl::visualization::PCL_VISUALIZER_POINT_SIZE, point_size);

		// update cloud in the viewer
		//viewer->updatePointCloud(cloudNative);
		viewer->updatePointCloud(cloudColoredBySegments);

		// update viewer
		viewer->spinOnce();

		// empty the cloud. new points will be added for next frame 
		cloudNative->clear();
		cloudColoredBySegments->clear();

		// display the current camera position in terminal to determine how to adjust the camera in the viewer
		/*pcl::visualization::Camera camera;

		viewer->getCameraParameters(camera);

		std::cout << "Camera position: ("
		<< camera.pos[0] << ", "
		<< camera.pos[1] << ", "
		<< camera.pos[2] << ", "
		<< camera.focal[0] << ", "
		<< camera.focal[1] << ", "
		<< camera.focal[2] << ", "
		<< camera.view[0] << ", "
		<< camera.view[1] << ", "
		<< camera.view[2] << ") " << std::endl;*/

		pHeliosDevice->RequeueBuffer(pHeliosImage);

	}

	pHeliosDevice->StopStream();
}






// -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
// -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
// -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
// -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
// -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------



Arena::DeviceInfo SelectDevice(std::vector<Arena::DeviceInfo>& deviceInfos)
{
	if (deviceInfos.size() == 1)
	{
		std::cout << "\n" << TAB1 << "Only one device detected: " << deviceInfos[0].ModelName() << TAB1 << deviceInfos[0].SerialNumber() << TAB1 << deviceInfos[0].IpAddressStr() << ".\n";
		std::cout << TAB1 << "Automatically selecting this device.\n";
		return deviceInfos[0];
	}

	std::cout << "\nSelect device:\n";
	for (size_t i = 0; i < deviceInfos.size(); i++)
	{
		std::cout << TAB1 << i + 1 << ". " << deviceInfos[i].ModelName() << TAB1 << deviceInfos[i].SerialNumber() << TAB1 << deviceInfos[i].IpAddressStr() << "\n";
	}
	size_t selection = 0;

	do
	{
		std::cout << TAB1 << "Make selection (1-" << deviceInfos.size() << "): ";
		std::cin >> selection;

		if (std::cin.fail())
		{
			std::cin.clear();
			while (std::cin.get() != '\n');
			std::cout << TAB1 << "Invalid input. Please enter a number.\n";
		}
		else if (selection <= 0 || selection > deviceInfos.size())
		{
			std::cout << TAB1 << "Invalid device selected. Please select a device in the range (1-" << deviceInfos.size() << ").\n";
		}

	} while (selection <= 0 || selection > deviceInfos.size());

	return deviceInfos[selection - 1];
}

// =-=-=-=-=-=-=-=-=-
// =- PREPARATION -=-
// =- & CLEAN UP =-=-
// =-=-=-=-=-=-=-=-=-

int main()
{
	// flag to track when an exception has been thrown
	bool exceptionThrown = false;

	std::cout << "Cpp_HTP_PCL_2_Segmentation";

	try
	{
		// prepare example
		Arena::ISystem* pSystem = Arena::OpenSystem();
		pSystem->UpdateDevices(100);
		std::vector<Arena::DeviceInfo> deviceInfos = pSystem->GetDevices();
		if (deviceInfos.size() == 0)
		{
			std::cout << "\nNo camera connected\nPress enter to complete\n";
			std::getchar();
			return 0;
		}
		Arena::DeviceInfo selectedDeviceInfo = SelectDevice(deviceInfos);
		Arena::IDevice* pDevice = pSystem->CreateDevice(selectedDeviceInfo);

		// enable stream auto negotiate packet size
		Arena::SetNodeValue<bool>(pDevice->GetTLStreamNodeMap(), "StreamAutoNegotiatePacketSize", true);

		// enable stream packet resend
		Arena::SetNodeValue<bool>(pDevice->GetTLStreamNodeMap(), "StreamPacketResendEnable", true);

		// enable flying pixels removal 
		Arena::SetNodeValue<bool>(pDevice->GetNodeMap(), "Scan3dFlyingPixelsRemovalEnable", true);

		// enable Standard HDR
		Arena::SetNodeValue<GenICam::gcstring>(pDevice->GetNodeMap(), "Scan3dHDRMode", "StandardHDR");

		std::cout << "Commence example\n\n";

		/*cv::Mat image_XYZ_mm;*/
		size_t width = 0;
		size_t height = 0;
		double scale;
		double offset_x;
		double offset_y;
		double offset_z;
		double offset_tof_to_xyz;

		pclExample(
			pDevice,
			width,
			height,
			scale,
			offset_x,
			offset_y,
			offset_z,
			offset_tof_to_xyz);

		std::cout << "\nExample complete\n";

		// clean up example
		pSystem->DestroyDevice(pDevice);
		Arena::CloseSystem(pSystem);
	}
	catch (GenICam::GenericException& ge)
	{
		std::cout << "\nGenICam exception thrown: " << ge.what() << "\n";
		exceptionThrown = true;
	}
	catch (std::exception& ex)
	{
		std::cout << "\nStandard exception thrown: " << ex.what() << "\n";
		exceptionThrown = true;
	}
	catch (...)
	{
		std::cout << "\nUnexpected exception thrown\n";
		exceptionThrown = true;
	}

	std::cout << "Press enter to complete\n";
	std::cin.ignore();
	std::getchar();

	if (exceptionThrown)
		return -1;
	else
		return 0;
}
